/*
 * Decompiled with CFR 0.152.
 */
package org.lovetropics.multimedia;

import java.io.Closeable;
import java.io.IOException;
import java.io.InputStream;
import java.nio.ByteBuffer;
import java.nio.channels.Channels;
import java.nio.channels.ReadableByteChannel;
import java.nio.channels.SeekableByteChannel;
import java.nio.file.Path;
import java.util.Objects;
import javax.sound.sampled.AudioFormat;
import org.jspecify.annotations.Nullable;
import org.lovetropics.multimedia.AudioDecoder;
import org.lovetropics.multimedia.AudioFrame;
import org.lovetropics.multimedia.AudioPacket;
import org.lovetropics.multimedia.DecoderException;
import org.lovetropics.multimedia.MultimediaNative;
import org.lovetropics.multimedia.MultimediaPacket;
import org.lovetropics.multimedia.Platform;
import org.lovetropics.multimedia.VideoDecoder;
import org.lovetropics.multimedia.VideoFrame;
import org.lovetropics.multimedia.VideoPacket;

public final class MultimediaReader
implements Closeable {
    private final long handle;
    private final double duration;
    private boolean closed;

    private MultimediaReader(long handle) {
        this.handle = handle;
        this.duration = MultimediaNative.getDuration(handle);
    }

    public static boolean isSupportedPlatform() {
        return Platform.tryDetect() != null;
    }

    public static MultimediaReader open(Path path) throws IOException {
        Objects.requireNonNull(path);
        String pathString = path.toAbsolutePath().normalize().toString();
        return new MultimediaReader(MultimediaNative.openPathReader(pathString));
    }

    public static MultimediaReader open(InputStream input) throws IOException {
        return MultimediaReader.open(Channels.newChannel(input));
    }

    public static MultimediaReader open(ReadableByteChannel channel) throws IOException {
        Objects.requireNonNull(channel);
        if (channel instanceof SeekableByteChannel) {
            SeekableByteChannel seekableChannel = (SeekableByteChannel)channel;
            return new MultimediaReader(MultimediaNative.openSeekableByteChannelReader(seekableChannel));
        }
        return new MultimediaReader(MultimediaNative.openByteChannelReader(channel));
    }

    private void checkOpen() {
        if (this.closed) {
            throw new IllegalStateException("Decoder has already been closed");
        }
    }

    public double duration() {
        return this.duration;
    }

    public synchronized @Nullable MultimediaPacket readPacket() throws IOException {
        this.checkOpen();
        long packet = MultimediaNative.readPacket(this.handle);
        if (packet == 0L) {
            return null;
        }
        int packetType = MultimediaNative.getPacketType(packet);
        return switch (packetType) {
            case 0 -> new VideoPacketImpl(packet);
            case 1 -> new AudioPacketImpl(packet);
            default -> {
                MultimediaNative.destroyPacket(packet);
                throw new IOException("Unrecognized packet type: " + packetType);
            }
        };
    }

    public synchronized @Nullable VideoDecoder openVideoDecoder() throws IOException, DecoderException {
        this.checkOpen();
        long videoDecoder = MultimediaNative.openVideoDecoder(this.handle);
        return videoDecoder != 0L ? new VideoDecoderImpl(videoDecoder) : null;
    }

    public synchronized @Nullable AudioDecoder openAudioDecoder(AudioFormat format) throws IOException, DecoderException {
        boolean stereo;
        this.checkOpen();
        int sampleFormat = MultimediaReader.getAudioSampleFormat(format.getEncoding(), format.getSampleSizeInBits());
        int channels = format.getChannels();
        if (channels == 1) {
            stereo = false;
        } else if (channels == 2) {
            stereo = true;
        } else {
            throw new IllegalArgumentException("Unsupported channels: " + channels);
        }
        int sampleRate = (int)format.getSampleRate();
        if ((float)sampleRate != format.getSampleRate() || format.getSampleRate() <= 0.0f) {
            throw new IllegalArgumentException("Sample rate must be a positive integer: " + format.getSampleRate());
        }
        long audioDecoder = MultimediaNative.openAudioDecoder(this.handle, sampleFormat, stereo, sampleRate);
        return audioDecoder != 0L ? new AudioDecoderImpl(audioDecoder, format) : null;
    }

    private static int getAudioSampleFormat(AudioFormat.Encoding encoding, int sampleBits) {
        if (sampleBits == 8 && encoding == AudioFormat.Encoding.PCM_UNSIGNED) {
            return 0;
        }
        if (sampleBits == 16 && encoding == AudioFormat.Encoding.PCM_SIGNED) {
            return 1;
        }
        if (sampleBits == 32) {
            if (encoding == AudioFormat.Encoding.PCM_SIGNED) {
                return 2;
            }
            if (encoding == AudioFormat.Encoding.PCM_FLOAT) {
                return 4;
            }
        }
        if (sampleBits == 64) {
            if (encoding == AudioFormat.Encoding.PCM_SIGNED) {
                return 3;
            }
            if (encoding == AudioFormat.Encoding.PCM_FLOAT) {
                return 5;
            }
        }
        throw new IllegalArgumentException("Unsupported encoding: " + String.valueOf(encoding) + " with " + sampleBits + " bits");
    }

    public synchronized void seekTo(double time) throws IOException {
        this.checkOpen();
        MultimediaNative.seekTo(this.handle, time);
    }

    @Override
    public synchronized void close() throws IOException {
        if (this.closed) {
            return;
        }
        this.closed = true;
        MultimediaNative.destroyReader(this.handle);
    }

    static final class VideoPacketImpl
    extends AbstractPacket
    implements VideoPacket {
        private VideoPacketImpl(long handle) {
            super(handle);
        }
    }

    static final class AudioPacketImpl
    extends AbstractPacket
    implements AudioPacket {
        private AudioPacketImpl(long handle) {
            super(handle);
        }
    }

    private static class VideoDecoderImpl
    implements VideoDecoder {
        private final long handle;
        private final int width;
        private final int height;
        private boolean maybeHasFrames;
        private boolean closed;

        private VideoDecoderImpl(long handle) {
            this.handle = handle;
            this.width = MultimediaNative.getVideoWidth(handle);
            this.height = MultimediaNative.getVideoHeight(handle);
        }

        @Override
        public int width() {
            return this.width;
        }

        @Override
        public int height() {
            return this.height;
        }

        private void checkOpen() {
            if (this.closed) {
                throw new IllegalStateException("Decoder has already been closed");
            }
        }

        private void checkCanSendPacket() {
            this.checkOpen();
            if (this.maybeHasFrames) {
                throw new IllegalStateException("Cannot send packet before all frames have been drained");
            }
        }

        @Override
        public void sendPacket(VideoPacket packet) throws DecoderException {
            this.checkCanSendPacket();
            try (VideoPacket videoPacket = packet;){
                MultimediaNative.sendVideoPacket(this.handle, ((VideoPacketImpl)packet).handle());
                this.maybeHasFrames = true;
            }
        }

        @Override
        public @Nullable VideoFrame readFrame() throws DecoderException {
            this.checkOpen();
            if (!this.maybeHasFrames) {
                return null;
            }
            long frameHandle = MultimediaNative.readVideoFrame(this.handle);
            if (frameHandle == 0L) {
                this.maybeHasFrames = false;
                return null;
            }
            return new VideoFrameImpl(frameHandle);
        }

        @Override
        public void close() {
            if (this.closed) {
                return;
            }
            this.closed = true;
            MultimediaNative.destroyVideoDecoder(this.handle);
        }
    }

    private static class AudioDecoderImpl
    implements AudioDecoder {
        private final long handle;
        private final AudioFormat format;
        private boolean maybeHasFrames;
        private boolean closed;

        private AudioDecoderImpl(long handle, AudioFormat format) {
            this.handle = handle;
            this.format = format;
        }

        private void checkOpen() {
            if (this.closed) {
                throw new IllegalStateException("Decoder has already been closed");
            }
        }

        private void checkCanSendPacket() {
            this.checkOpen();
            if (this.maybeHasFrames) {
                throw new IllegalStateException("Cannot send packet before all frames have been drained");
            }
        }

        @Override
        public AudioFormat format() {
            return this.format;
        }

        @Override
        public void sendPacket(AudioPacket packet) throws DecoderException {
            this.checkCanSendPacket();
            try (AudioPacket audioPacket = packet;){
                MultimediaNative.sendAudioPacket(this.handle, ((AudioPacketImpl)packet).handle());
                this.maybeHasFrames = true;
            }
        }

        @Override
        public @Nullable AudioFrame readFrame() throws DecoderException {
            this.checkOpen();
            if (!this.maybeHasFrames) {
                return null;
            }
            long frameHandle = MultimediaNative.readAudioFrame(this.handle);
            if (frameHandle == 0L) {
                this.maybeHasFrames = false;
                return null;
            }
            return new AudioFrameImpl(frameHandle);
        }

        @Override
        public void close() {
            if (this.closed) {
                return;
            }
            this.closed = true;
            MultimediaNative.destroyAudioDecoder(this.handle);
        }
    }

    private static abstract class AbstractPacket
    implements AutoCloseable {
        private final long handle;
        private boolean closed;

        protected AbstractPacket(long handle) {
            this.handle = handle;
        }

        public long handle() {
            if (this.closed) {
                throw new IllegalStateException("Packet has already been freed");
            }
            return this.handle;
        }

        @Override
        public void close() {
            if (this.closed) {
                return;
            }
            this.closed = true;
            MultimediaNative.destroyPacket(this.handle);
        }
    }

    private static class AudioFrameImpl
    implements AudioFrame {
        private final long handle;
        private final double presentTime;
        private final int samples;
        private final int bytes;
        private boolean closed;

        private AudioFrameImpl(long handle) {
            this.handle = handle;
            this.presentTime = MultimediaNative.getAudioFramePresentTime(handle);
            this.samples = MultimediaNative.getAudioFrameSamples(handle);
            this.bytes = MultimediaNative.getAudioFrameBytes(handle);
        }

        @Override
        public double presentTime() {
            return this.presentTime;
        }

        @Override
        public int samples() {
            return this.samples;
        }

        @Override
        public int bytes() {
            return this.bytes;
        }

        @Override
        public void unpackSamples(ByteBuffer output) throws DecoderException {
            if (this.closed) {
                throw new IllegalStateException("Frame has already been closed");
            }
            this.closed = true;
            if (output.remaining() < this.bytes) {
                throw new IllegalArgumentException("Output buffer is too small, must have at least " + this.bytes + " remaining but had " + output.remaining());
            }
            MultimediaNative.unpackAudioSamples(this.handle, output, output.position());
            output.position(output.position() + this.bytes);
        }

        @Override
        public void close() {
            if (this.closed) {
                return;
            }
            this.closed = true;
            MultimediaNative.destroyAudioFrame(this.handle);
        }
    }

    private static class VideoFrameImpl
    implements VideoFrame {
        private final long handle;
        private final double presentTime;
        private final double presentEndTime;
        private boolean closed;

        private VideoFrameImpl(long handle) {
            this.handle = handle;
            this.presentTime = MultimediaNative.getVideoFramePresentTime(handle);
            this.presentEndTime = MultimediaNative.getVideoFramePresentEndTime(handle);
        }

        @Override
        public double presentTime() {
            return this.presentTime;
        }

        @Override
        public double presentEndTime() {
            return this.presentEndTime;
        }

        @Override
        public void unpackPixels(int outputWidth, int outputHeight, ByteBuffer output) throws DecoderException {
            if (this.closed) {
                throw new IllegalStateException("Frame has already been closed");
            }
            this.closed = true;
            if (outputWidth <= 0 || outputHeight <= 0) {
                throw new IllegalArgumentException("Output width and height must be positive");
            }
            int bytes = outputWidth * outputHeight * 4;
            if (output.remaining() < bytes) {
                throw new IllegalArgumentException("Output buffer is too small, must have at least " + bytes + " remaining but had " + output.remaining());
            }
            MultimediaNative.unpackVideoPixels(this.handle, outputWidth, outputHeight, output, output.position());
            output.position(output.position() + bytes);
        }

        @Override
        public void close() {
            if (this.closed) {
                return;
            }
            this.closed = true;
            MultimediaNative.destroyVideoFrame(this.handle);
        }
    }
}

