/*
 * Decompiled with CFR 0.152.
 */
package org.lovetropics.multimedia.mod.client.cache;

import com.mojang.blaze3d.platform.NativeImage;
import com.mojang.logging.LogUtils;
import java.io.Closeable;
import java.io.IOException;
import java.lang.runtime.SwitchBootstraps;
import java.net.URI;
import java.net.http.HttpClient;
import java.net.http.HttpRequest;
import java.net.http.HttpResponse;
import java.nio.ByteBuffer;
import java.nio.channels.SeekableByteChannel;
import java.nio.file.FileVisitResult;
import java.nio.file.FileVisitor;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.FileAttribute;
import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionException;
import java.util.concurrent.Executor;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import net.minecraft.FileUtil;
import net.minecraft.Util;
import net.minecraft.util.thread.ConsecutiveExecutor;
import org.apache.commons.io.FilenameUtils;
import org.apache.commons.io.IOUtils;
import org.lovetropics.multimedia.MultimediaReader;
import org.lovetropics.multimedia.mod.MediaFile;
import org.lovetropics.multimedia.mod.client.cache.CachedFile;
import org.lovetropics.multimedia.mod.client.cache.FileDownload;
import org.lovetropics.multimedia.mod.client.cache.MediaFileId;
import org.lovetropics.multimedia.mod.client.cache.MediaIndex;
import org.lwjgl.BufferUtils;
import org.slf4j.Logger;

public class MediaFileCache {
    private static final Logger LOGGER = LogUtils.getLogger();
    private static final Duration ALWAYS_FRESH_BEFORE = Duration.ofMinutes(5L);
    private static final Duration ALWAYS_CHECK_AFTER = Duration.ofHours(1L);
    private static final Duration EXPIRE_AFTER_ACCESS = Duration.ofDays(2L);
    private final Path rootPath;
    private final String userAgent;
    private final Path indexPath;
    private final HttpClient httpClient = HttpClient.newBuilder().executor((Executor)Util.nonCriticalIoPool()).build();
    private final ConsecutiveExecutor consecutiveExecutor = new ConsecutiveExecutor((Executor)Util.ioPool(), "lt-media-cache");
    private final List<CachedFile> files = new ArrayList<CachedFile>();
    private final Map<URI, CompletableFuture<CachedFile>> pendingRequests = new HashMap<URI, CompletableFuture<CachedFile>>();

    public MediaFileCache(Path rootPath, String userAgent) {
        this.rootPath = rootPath;
        this.indexPath = rootPath.resolve("index.json");
        this.userAgent = userAgent;
        this.consecutiveExecutor.schedule(() -> {
            try {
                Files.createDirectories(rootPath, new FileAttribute[0]);
            }
            catch (IOException e) {
                LOGGER.error("Failed to create cache directory: {}", (Object)rootPath, (Object)e);
            }
            MediaIndex index = MediaIndex.load(this.indexPath);
            for (MediaIndex.CachedFile indexFile : index.cachedFiles()) {
                CachedFile file = CachedFile.fromCache(rootPath, indexFile);
                if (file == null) continue;
                this.files.add(file);
            }
            this.cleanCache();
        });
    }

    private void cleanCache() {
        this.files.removeIf(file -> file.getTimeSinceLastAccess().compareTo(EXPIRE_AFTER_ACCESS) > 0);
        this.cleanOldVersions();
        this.storeIndex();
        this.deleteUnreferencedFiles();
    }

    private void cleanOldVersions() {
        Map<URI, List<CachedFile>> filesByUri = this.files.stream().collect(Collectors.groupingBy(file -> file.fileId().uri()));
        filesByUri.forEach((uri, files) -> {
            if (files.size() < 2) {
                return;
            }
            CachedFile latestFile = files.stream().max(Comparator.comparing(f -> f.fileId().resolvedAt())).orElse(null);
            for (CachedFile file : files) {
                if (file == latestFile) continue;
                this.files.remove(file);
            }
        });
    }

    private void deleteUnreferencedFiles() {
        HashSet<Path> knownPaths = new HashSet<Path>();
        knownPaths.add(this.indexPath);
        for (CachedFile file : this.files) {
            knownPaths.add(file.path());
        }
        try {
            MediaFileCache.deleteAllExcept(this.rootPath, knownPaths);
        }
        catch (IOException e) {
            LOGGER.error("Failed to delete cache files", (Throwable)e);
        }
    }

    private static void deleteAllExcept(final Path rootPath, final Set<Path> knownFiles) throws IOException {
        Files.walkFileTree(rootPath, (FileVisitor<? super Path>)new SimpleFileVisitor<Path>(){

            @Override
            public FileVisitResult preVisitDirectory(Path dir, BasicFileAttributes attrs) {
                return dir.equals(rootPath) ? FileVisitResult.CONTINUE : FileVisitResult.SKIP_SUBTREE;
            }

            @Override
            public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
                if (!knownFiles.contains(file)) {
                    Files.delete(file);
                }
                return FileVisitResult.CONTINUE;
            }
        });
    }

    public CompletableFuture<Void> ensureDownloaded(MediaFile file) {
        if (file.isLocalFile()) {
            return CompletableFuture.completedFuture(null);
        }
        return this.getOrDownload(file.uri()).thenCompose(CachedFile::awaitDownload);
    }

    public MultimediaReader openMultimediaReader(MediaFile file) throws IOException {
        SeekableByteChannel channel = this.openChannel(file);
        try {
            return MultimediaReader.open(channel);
        }
        catch (IOException e) {
            IOUtils.closeQuietly((Closeable)channel);
            throw e;
        }
    }

    public NativeImage loadImage(MediaFile file) throws IOException {
        try (SeekableByteChannel channel = this.openChannel(file);){
            ByteBuffer buffer = BufferUtils.createByteBuffer((int)Math.toIntExact(channel.size()));
            while (buffer.hasRemaining()) {
                channel.read(buffer);
            }
            buffer.flip();
            NativeImage nativeImage = NativeImage.read((ByteBuffer)buffer);
            return nativeImage;
        }
    }

    public SeekableByteChannel openChannel(MediaFile file) throws IOException {
        CachedFile cached;
        if (file.isLocalFile() && MediaFile.CAN_PLAY_FROM_LOCAL_FILE) {
            return Files.newByteChannel(Path.of(file.uri()), new OpenOption[0]);
        }
        try {
            cached = this.getOrDownload(file.uri()).join();
        }
        catch (CompletionException e) {
            Throwable throwable = e.getCause();
            Objects.requireNonNull(throwable);
            Throwable throwable2 = throwable;
            int n = 0;
            switch (SwitchBootstraps.typeSwitch("typeSwitch", new Object[]{IOException.class, Throwable.class}, (Object)throwable2, n)) {
                case 0: {
                    IOException cause = (IOException)throwable2;
                    throw cause;
                }
            }
            Throwable cause = throwable2;
            throw new IOException(cause);
        }
        return cached.openChannel();
    }

    private CompletableFuture<CachedFile> getOrDownload(URI uri) {
        return CompletableFuture.supplyAsync(() -> {
            CompletableFuture<CachedFile> pendingRequest = this.pendingRequests.get(uri);
            if (pendingRequest != null) {
                return pendingRequest;
            }
            CompletableFuture<CachedFile> future = this.getOrDownloadInternal(uri);
            this.pendingRequests.put(uri, future);
            future.thenRunAsync(() -> this.pendingRequests.remove(uri, future), arg_0 -> ((ConsecutiveExecutor)this.consecutiveExecutor).schedule(arg_0));
            return future;
        }, arg_0 -> ((ConsecutiveExecutor)this.consecutiveExecutor).schedule(arg_0)).thenCompose(Function.identity());
    }

    private CompletableFuture<CachedFile> getOrDownloadInternal(URI uri) {
        List<CachedFile> candidateFiles = this.streamFilesForUri(uri).toList();
        if (candidateFiles.isEmpty()) {
            return this.requestDownload(uri, List.of());
        }
        CachedFile latestFile = candidateFiles.stream().max(Comparator.comparing(f -> f.fileId().resolvedAt())).get();
        Duration latestFileAge = latestFile.fileId().getCurrentAge();
        if (latestFileAge.compareTo(ALWAYS_FRESH_BEFORE) <= 0) {
            return this.useCachedFile(latestFile);
        }
        List<String> candidateEtags = candidateFiles.stream().flatMap(cached -> cached.fileId().etag().stream()).toList();
        if (candidateEtags.isEmpty() && latestFileAge.compareTo(ALWAYS_CHECK_AFTER) <= 0) {
            return this.useCachedFile(latestFile);
        }
        return this.requestDownload(uri, candidateEtags);
    }

    private Stream<CachedFile> streamFilesForUri(URI uri) {
        return this.files.stream().filter(cached -> cached.fileId().uri().equals(uri));
    }

    private CompletableFuture<CachedFile> useCachedFile(CachedFile file) {
        file.touch();
        this.storeIndex();
        return CompletableFuture.completedFuture(file);
    }

    private CompletableFuture<CachedFile> requestDownload(URI uri, List<String> ifNoneMatch) {
        HttpRequest.Builder request = HttpRequest.newBuilder(uri).GET().header("User-Agent", this.userAgent);
        if (!ifNoneMatch.isEmpty()) {
            request.header("If-None-Match", String.join((CharSequence)", ", ifNoneMatch));
        }
        return this.httpClient.sendAsync(request.build(), FileDownload.bodyHandler()).thenApplyAsync(response -> {
            try {
                return this.handleDownloadResponse(uri, (HttpResponse<FileDownload.Response>)response);
            }
            catch (IOException e) {
                throw new CompletionException(e);
            }
        }, arg_0 -> ((ConsecutiveExecutor)this.consecutiveExecutor).schedule(arg_0));
    }

    private CachedFile handleDownloadResponse(URI uri, HttpResponse<FileDownload.Response> response) throws IOException {
        Optional<String> etag = response.headers().firstValue("ETag");
        if (response.statusCode() == 304) {
            CachedFile matchingFile = this.streamFilesForUri(uri).filter(cached -> cached.fileId().etag().equals(etag)).findFirst().orElseThrow(() -> new IOException("Got NOT_MODIFIED, but ETag (" + String.valueOf(etag) + ") did not match any file we have locally"));
            matchingFile.touch();
            this.storeIndex();
            return matchingFile;
        }
        if (response.statusCode() < 200 || response.statusCode() >= 300) {
            throw new IOException("Unexpected status code " + response.statusCode());
        }
        return this.startDownload(uri, etag, response.body());
    }

    private CachedFile startDownload(URI uri, Optional<String> etag, FileDownload.Response body) throws IOException {
        Path path = this.selectFilePath(uri);
        FileDownload download = body.startWritingTo(path);
        MediaFileId fileId = new MediaFileId(uri, etag, download.size(), Instant.now());
        CachedFile file = CachedFile.fromDownload(fileId, download);
        this.files.add(file);
        file.awaitDownload().thenRunAsync(this::storeIndex, arg_0 -> ((ConsecutiveExecutor)this.consecutiveExecutor).schedule(arg_0));
        return file;
    }

    private Path selectFilePath(URI uri) {
        Path newPath;
        String fileName = FilenameUtils.getName((String)uri.getPath());
        if (fileName.isBlank()) {
            fileName = "media";
        }
        String baseName = FileUtil.sanitizeName((String)FilenameUtils.removeExtension((String)fileName));
        String extension = FilenameUtils.getExtension((String)fileName);
        int index = 1;
        do {
            String newFileName = baseName + "_" + index;
            if (!extension.isEmpty()) {
                newFileName = newFileName + "." + extension;
            }
            newPath = this.rootPath.resolve(newFileName);
            ++index;
        } while (Files.exists(newPath, new LinkOption[0]));
        return newPath;
    }

    private void storeIndex() {
        List<MediaIndex.CachedFile> downloadedFiles = this.files.stream().filter(CachedFile::isDownloaded).map(file -> file.asIndexFile(this.rootPath)).toList();
        MediaIndex index = new MediaIndex(downloadedFiles);
        index.store(this.indexPath);
    }
}

