/*
 * Decompiled with CFR 0.152.
 */
package org.lovetropics.multimedia.mod.client.playback;

import com.mojang.blaze3d.audio.Channel;
import com.mojang.blaze3d.audio.ListenerTransform;
import com.mojang.blaze3d.audio.OpenAlUtil;
import com.mojang.blaze3d.audio.SoundBuffer;
import com.mojang.logging.LogUtils;
import java.nio.ByteBuffer;
import java.util.ArrayDeque;
import java.util.Deque;
import java.util.Objects;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import net.minecraft.client.sounds.ChannelAccess;
import net.minecraft.client.sounds.SoundManager;
import net.minecraft.util.Mth;
import net.minecraft.world.phys.Vec3;
import org.jetbrains.annotations.Nullable;
import org.lovetropics.multimedia.AudioDecoder;
import org.lovetropics.multimedia.AudioFrame;
import org.lovetropics.multimedia.AudioPacket;
import org.lovetropics.multimedia.DecoderException;
import org.lovetropics.multimedia.mod.client.playback.AudioSampler;
import org.lovetropics.multimedia.mod.client.playback.AudioWorldSource;
import org.lovetropics.multimedia.mod.client.playback.ClockSyncer;
import org.lovetropics.multimedia.mod.client.playback.PacketReader;
import org.lwjgl.openal.AL10;
import org.slf4j.Logger;

public class AudioPlaybackChannel
extends Channel {
    private static final Logger LOGGER = LogUtils.getLogger();
    private static final double QUEUE_AT_LEAST_SECONDS = 0.25;
    private static final double MAX_CLOCK_DRIFT = 0.01;
    private static final double CORRECT_OVER_SECONDS = 5.0;
    private final SoundManager soundManager;
    private final PacketReader packets;
    private final AudioDecoder decoder;
    private final ClockSyncer clock;
    private final AudioSampler sampler;
    private final Deque<QueuedFrame> queuedFrames = new ArrayDeque<QueuedFrame>();
    private double lastPlayedFrameEndTime;
    @Nullable
    private ClockCorrection activeCorrection;
    @Nullable
    private AudioWorldSource worldSource;
    private volatile boolean seeking;
    private volatile boolean closed;

    private AudioPlaybackChannel(int source, SoundManager soundManager, PacketReader packets, AudioDecoder decoder, ClockSyncer clock) {
        super(source);
        this.soundManager = soundManager;
        this.packets = packets;
        this.decoder = decoder;
        this.clock = clock;
        this.sampler = new AudioSampler(decoder.format());
        this.setupSource();
    }

    private void setupSource() {
        AL10.alSourcef((int)this.source, (int)4110, (float)10.0f);
        this.setVolume(1.0f);
        this.setPitch(1.0f);
        this.disableAttenuation();
        this.setLooping(false);
        this.setSelfPosition(Vec3.ZERO);
        this.setRelative(true);
    }

    public static Handle create(SoundManager soundManager, PacketReader packets, AudioDecoder decoder, ClockSyncer clock) {
        ChannelAccess channelAccess = soundManager.soundEngine.channelAccess;
        CompletableFuture<AudioPlaybackChannel> future = CompletableFuture.supplyAsync(() -> {
            int[] sources = new int[1];
            AL10.alGenSources((int[])sources);
            if (OpenAlUtil.checkALError((String)"Playback audio source")) {
                return null;
            }
            AudioPlaybackChannel channel = new AudioPlaybackChannel(sources[0], soundManager, packets, decoder, clock);
            channelAccess.channels.add(AudioPlaybackChannel.createChannelHandle(channelAccess, channel));
            return channel;
        }, channelAccess.executor);
        return new Handle(channelAccess, future);
    }

    private static ChannelAccess.ChannelHandle createChannelHandle(final ChannelAccess channelAccess, final Channel channel) {
        ChannelAccess channelAccess2 = channelAccess;
        Objects.requireNonNull(channelAccess2);
        return new ChannelAccess.ChannelHandle(channelAccess2, channel){
            private boolean destroyed;
            {
                ChannelAccess channelAccess2 = x0;
                Objects.requireNonNull(channelAccess2);
                super(channelAccess2, arg0);
            }

            public boolean isStopped() {
                return this.destroyed;
            }

            public void execute(Consumer<Channel> consumer) {
                channelAccess.executor.execute(() -> {
                    if (!this.destroyed) {
                        consumer.accept(channel);
                    }
                });
            }

            public void release() {
                this.destroyed = true;
                channel.destroy();
            }
        };
    }

    public void setWorldSource(AudioWorldSource worldSource) {
        this.worldSource = worldSource;
        ListenerTransform listenerTransform = this.soundManager.getListenerTransform();
        this.linearAttenuation(worldSource.attenuationDistance());
        this.setSelfPosition(worldSource.resolveSourcePos(listenerTransform));
        this.setRelative(false);
    }

    private void reset() {
        if (AL10.alGetSourcei((int)this.source, (int)4112) == 4113) {
            this.play();
        }
        this.stop();
        this.unqueueBuffers(this.queuedFrames.size());
        this.queuedFrames.clear();
        this.lastPlayedFrameEndTime = 0.0;
        this.activeCorrection = null;
    }

    public void endSeek() {
        if (!this.seeking) {
            throw new IllegalStateException("Not seeking");
        }
        this.seeking = false;
        this.lastPlayedFrameEndTime = this.clock.getElapsedTime();
        this.updateStream();
    }

    public boolean stopped() {
        return this.closed;
    }

    public void updateStream() {
        if (this.closed || this.seeking) {
            return;
        }
        if (this.worldSource != null) {
            ListenerTransform listenerTransform = this.soundManager.getListenerTransform();
            this.setSelfPosition(this.worldSource.resolveSourcePos(listenerTransform));
        }
        boolean stopped = AL10.alGetSourcei((int)this.source, (int)4112) == 4116;
        int processedBuffers = stopped && this.clock.isPaused() ? 0 : this.removeProcessedBuffers();
        for (int i = 0; i < processedBuffers; ++i) {
            this.lastPlayedFrameEndTime = this.queuedFrames.removeFirst().endTime();
        }
        if (this.playing() && this.clock.isPaused()) {
            this.pause();
        }
        double speedFactor = this.syncClocks(this.getAudioClockTime());
        this.tryQueueFrames(this.lastPlayedFrameEndTime + 0.25, speedFactor);
        if (!(this.playing() || this.clock.isPaused() || this.queuedFrames.isEmpty())) {
            this.play();
        }
    }

    private double getAudioClockTime() {
        if (this.queuedFrames.isEmpty()) {
            return this.lastPlayedFrameEndTime;
        }
        double currentSpeedFactor = this.queuedFrames.peekFirst().speedFactor();
        return this.lastPlayedFrameEndTime + (double)AL10.alGetSourcef((int)this.source, (int)4132) / currentSpeedFactor;
    }

    private double syncClocks(double audioClockTime) {
        if (this.queuedFrames.isEmpty() && this.packets.isEof()) {
            return 1.0;
        }
        if (this.clock.requestSyncTo(audioClockTime)) {
            return 1.0;
        }
        double clockTime = this.clock.getElapsedTime();
        double clockDrift = clockTime - audioClockTime;
        if (this.activeCorrection != null && (clockTime > this.activeCorrection.finishAt || Math.abs(clockDrift) < 0.01)) {
            this.activeCorrection = null;
        }
        if (this.activeCorrection == null && Math.abs(clockDrift) > 0.01) {
            this.activeCorrection = new ClockCorrection(Mth.clamp((double)(1.0 + clockDrift / 5.0), (double)0.5, (double)2.0), clockTime + 2.5);
        }
        return this.activeCorrection != null ? this.activeCorrection.speedFactor : 1.0;
    }

    private void tryQueueFrames(double queueUntilTime, double speedFactor) {
        try {
            this.queueFrames(queueUntilTime, speedFactor);
        }
        catch (DecoderException e) {
            LOGGER.error("Failed to read from audio stream", (Throwable)e);
            this.scheduleClose();
        }
    }

    private void queueFrames(double queueUntilTime, double speedFactor) throws DecoderException {
        while (this.queuedFrames.isEmpty() || this.queuedFrames.getLast().endTime() < queueUntilTime) {
            AudioFrame frame = this.decoder.readFrame();
            if (frame != null) {
                this.queueFrame(frame, speedFactor);
                continue;
            }
            AudioPacket packet = this.packets.pollAudioPacket();
            if (packet == null) break;
            AudioPacket audioPacket = packet;
            try {
                this.decoder.sendPacket(packet);
            }
            finally {
                if (audioPacket == null) continue;
                audioPacket.close();
            }
        }
    }

    private void queueFrame(AudioFrame frame, double speedFactor) throws DecoderException {
        try (AudioFrame audioFrame = frame;){
            double frameDuration = (double)frame.samples() / (double)this.decoder.format().getSampleRate();
            double frameEndTime = frame.presentTime() + frameDuration;
            ByteBuffer buffer = this.sampler.sample(frame, Mth.floor((double)((double)frame.samples() / speedFactor)));
            new SoundBuffer(buffer, this.decoder.format()).releaseAlBuffer().ifPresent(id -> {
                AL10.alSourceQueueBuffers((int)this.source, (int[])new int[]{id});
                this.queuedFrames.addLast(new QueuedFrame(frameEndTime, speedFactor));
            });
        }
    }

    private int removeProcessedBuffers() {
        int bufferCount = AL10.alGetSourcei((int)this.source, (int)4118);
        this.unqueueBuffers(bufferCount);
        return bufferCount;
    }

    private void unqueueBuffers(int bufferCount) {
        if (bufferCount == 0) {
            return;
        }
        int[] buffers = new int[bufferCount];
        AL10.alSourceUnqueueBuffers((int)this.source, (int[])buffers);
        OpenAlUtil.checkALError((String)"Unqueue buffers");
        AL10.alDeleteBuffers((int[])buffers);
        OpenAlUtil.checkALError((String)"Remove processed buffers");
    }

    public void scheduleClose() {
        if (this.closed) {
            return;
        }
        this.packets.discardAudio();
        this.stop();
        this.closed = true;
    }

    public void destroy() {
        this.reset();
        super.destroy();
        this.decoder.close();
    }

    static class Handle
    implements AutoCloseable {
        private final ChannelAccess channelAccess;
        private final CompletableFuture<AudioPlaybackChannel> inner;

        private Handle(ChannelAccess channelAccess, CompletableFuture<AudioPlaybackChannel> inner) {
            this.channelAccess = channelAccess;
            this.inner = inner;
        }

        private void execute(Consumer<AudioPlaybackChannel> consumer) {
            this.inner.thenAcceptAsync((Consumer)consumer, this.channelAccess.executor);
        }

        public void setVolume(float volume) {
            this.execute(channel -> channel.setVolume(volume));
        }

        public void setWorldSource(AudioWorldSource source) {
            this.execute(channel -> channel.setWorldSource(source));
        }

        public void beginSeek() {
            this.inner.join().seeking = true;
            this.execute(AudioPlaybackChannel::reset);
        }

        public void endSeek() {
            this.execute(AudioPlaybackChannel::endSeek);
        }

        @Override
        public void close() {
            this.execute(AudioPlaybackChannel::scheduleClose);
        }
    }

    private record ClockCorrection(double speedFactor, double finishAt) {
    }

    private record QueuedFrame(double endTime, double speedFactor) {
    }
}

