/*
 * Decompiled with CFR 0.152.
 */
package org.lovetropics.multimedia.mod.client.playback;

import com.mojang.blaze3d.buffers.GpuBuffer;
import com.mojang.blaze3d.buffers.GpuFence;
import com.mojang.blaze3d.platform.NativeImage;
import com.mojang.blaze3d.systems.GpuDevice;
import com.mojang.blaze3d.textures.GpuTexture;
import com.mojang.blaze3d.textures.TextureFormat;
import java.util.Objects;
import java.util.concurrent.atomic.AtomicReference;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.ReentrantLock;
import javax.annotation.Nullable;
import org.lovetropics.multimedia.DecoderException;
import org.lovetropics.multimedia.VideoFrame;
import org.lovetropics.multimedia.mod.client.GpuExtensions;
import org.lovetropics.multimedia.mod.client.playback.ClockSyncer;
import org.lovetropics.multimedia.mod.client.playback.FrameSize;
import org.lovetropics.multimedia.mod.client.playback.PresentableVideoFrame;

public class VideoFrameUploader
implements AutoCloseable {
    private static final int BUFFER_COUNT = 3;
    private final GpuDevice device;
    private FrameSize requestedFrameSize;
    private final ClockSyncer clock;
    private final FrameBuffer[] frameBuffers = new FrameBuffer[3];
    private int nextWriteIndex;
    private int nextReadIndex;
    private final ReentrantLock writeFrameLock = new ReentrantLock();
    private final Condition canWrite = this.writeFrameLock.newCondition();
    private boolean seeking;

    public VideoFrameUploader(GpuDevice device, FrameSize frameSize, ClockSyncer clock) {
        this.device = device;
        this.requestedFrameSize = frameSize;
        for (int i = 0; i < this.frameBuffers.length; ++i) {
            this.frameBuffers[i] = new FrameBuffer(device, frameSize);
        }
        this.clock = clock;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void beginSeek() {
        this.writeFrameLock.lock();
        try {
            if (this.seeking) {
                return;
            }
            this.seeking = true;
            this.nextWriteIndex = 0;
            this.nextReadIndex = 0;
            for (FrameBuffer buffer : this.frameBuffers) {
                buffer.discardAndFreeze(this.device, this.requestedFrameSize);
            }
        }
        finally {
            this.writeFrameLock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void endSeek() {
        this.writeFrameLock.lock();
        try {
            if (!this.seeking) {
                throw new IllegalStateException("Not seeking");
            }
            this.seeking = false;
            for (FrameBuffer buffer : this.frameBuffers) {
                buffer.unfreeze();
            }
        }
        finally {
            this.writeFrameLock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void tick() {
        boolean readyForWrite = false;
        for (FrameBuffer buffer : this.frameBuffers) {
            buffer.tick(this.device, this.requestedFrameSize);
            readyForWrite |= buffer.isReadyForWrite();
        }
        this.discardExpiredFrames();
        if (readyForWrite) {
            this.writeFrameLock.lock();
            try {
                this.canWrite.signal();
            }
            finally {
                this.writeFrameLock.unlock();
            }
        }
    }

    private void discardExpiredFrames() {
        int index;
        int nextIndex;
        FrameBuffer nextFrameBuffer;
        for (int i = 0; i < this.frameBuffers.length - 1 && (nextFrameBuffer = this.frameBuffers[nextIndex = ((index = (this.nextReadIndex + i) % this.frameBuffers.length) + 1) % this.frameBuffers.length]).isReadyToPresent() && this.clock.getElapsedTime() >= nextFrameBuffer.presentTime; ++i) {
            this.frameBuffers[index].recycle(this.device, this.requestedFrameSize);
            this.nextReadIndex = nextIndex;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void writeFrame(VideoFrame frame) throws DecoderException, InterruptedException {
        if (this.clock.getElapsedTime() > frame.presentEndTime()) {
            frame.close();
            return;
        }
        this.writeFrameLock.lock();
        try (VideoFrame videoFrame = frame;){
            if (this.seeking) {
                return;
            }
            while (!this.frameBuffers[this.nextWriteIndex].tryWriteFrom(frame)) {
                this.canWrite.await();
                if (!this.seeking) continue;
                return;
            }
            this.nextWriteIndex = (this.nextWriteIndex + 1) % this.frameBuffers.length;
        }
        finally {
            this.writeFrameLock.unlock();
        }
    }

    @Nullable
    public PresentableVideoFrame takeNextFrame() {
        if (this.seeking) {
            return null;
        }
        final FrameBuffer buffer = this.frameBuffers[this.nextReadIndex];
        if (!buffer.isReadyToPresent() || this.clock.getElapsedTime() < buffer.presentTime) {
            return null;
        }
        return new PresentableVideoFrame(){
            private boolean closed;

            private void checkValid() {
                if (this.closed) {
                    throw new IllegalStateException("Frame has already been closed");
                }
            }

            @Override
            public void copyTo(GpuTexture texture) {
                this.checkValid();
                buffer.copyTo(VideoFrameUploader.this.device, texture);
            }

            @Override
            public FrameSize frameSize() {
                this.checkValid();
                return buffer.frameSize;
            }

            @Override
            public void close() {
                if (this.closed) {
                    return;
                }
                this.closed = true;
                buffer.recycle(VideoFrameUploader.this.device, VideoFrameUploader.this.requestedFrameSize);
                VideoFrameUploader.this.nextReadIndex = (VideoFrameUploader.this.nextReadIndex + 1) % VideoFrameUploader.this.frameBuffers.length;
            }
        };
    }

    public void requestFrameSize(FrameSize frameSize) {
        this.requestedFrameSize = frameSize;
    }

    @Override
    public void close() {
        for (FrameBuffer buffer : this.frameBuffers) {
            buffer.close();
        }
    }

    private static class FrameBuffer
    implements AutoCloseable {
        private GpuBuffer buffer;
        private volatile FrameSize frameSize;
        @Nullable
        private volatile GpuBuffer.MappedView mappedView;
        private volatile double presentTime;
        private final AtomicReference<State> state = new AtomicReference<State>(State.READY_FOR_WRITE);
        @Nullable
        private GpuFence recycleFence;

        private FrameBuffer(GpuDevice device, FrameSize frameSize) {
            this.frameSize = frameSize;
            this.buffer = this.createBuffer(device, frameSize);
            this.mapBuffer(device, frameSize);
        }

        private boolean tryUpdateState(State fromState, State toState) {
            return this.state.compareAndSet(fromState, toState);
        }

        private void updateState(State fromState, State toState) {
            State oldState = this.state.compareAndExchange(fromState, toState);
            if (oldState != fromState) {
                throw new IllegalStateException("Expected state " + String.valueOf((Object)fromState) + ", but was " + String.valueOf((Object)oldState));
            }
        }

        private void checkState(State expectedState) {
            State state = this.state.get();
            if (state != expectedState) {
                throw new IllegalStateException("Expected state " + String.valueOf((Object)expectedState) + ", but was " + String.valueOf((Object)state));
            }
        }

        private GpuBuffer createBuffer(GpuDevice device, FrameSize frameSize) {
            int sizeBytes = frameSize.width() * frameSize.height() * TextureFormat.RGBA8.pixelSize();
            return device.createBuffer(() -> "Video frame buffer", 18, sizeBytes);
        }

        private void mapBuffer(GpuDevice device, FrameSize requestedFrameSize) {
            if (this.mappedView != null) {
                throw new IllegalStateException("Buffer is already mapped");
            }
            if (!this.frameSize.equals(requestedFrameSize)) {
                this.buffer.close();
                this.buffer = this.createBuffer(device, requestedFrameSize);
                this.frameSize = requestedFrameSize;
            }
            this.mappedView = device.createCommandEncoder().mapBuffer(this.buffer, false, true);
        }

        public void tick(GpuDevice device, FrameSize requestedFrameSize) {
            if (this.tryUpdateState(State.WRITE_END, State.READY_TO_PRESENT)) {
                Objects.requireNonNull(this.mappedView).close();
                this.mappedView = null;
            } else {
                this.tickRecycle(device, requestedFrameSize);
            }
        }

        private void tickRecycle(GpuDevice device, FrameSize requestedFrameSize) {
            if (this.state.get() != State.RECYCLING) {
                return;
            }
            if (this.recycleFence == null || this.recycleFence.awaitCompletion(0L)) {
                if (this.recycleFence != null) {
                    this.recycleFence.close();
                    this.recycleFence = null;
                }
                this.mapBuffer(device, requestedFrameSize);
                this.updateState(State.RECYCLING, State.READY_FOR_WRITE);
            }
        }

        public boolean tryWriteFrom(VideoFrame frame) throws DecoderException {
            if (!this.tryUpdateState(State.READY_FOR_WRITE, State.WRITE_BEGIN)) {
                return false;
            }
            GpuBuffer.MappedView mappedView = Objects.requireNonNull(this.mappedView);
            this.presentTime = frame.presentTime();
            frame.unpackPixels(this.frameSize.width(), this.frameSize.height(), mappedView.data());
            mappedView.data().flip();
            this.updateState(State.WRITE_BEGIN, State.WRITE_END);
            return true;
        }

        public boolean isReadyForWrite() {
            return this.state.get() == State.READY_FOR_WRITE;
        }

        public boolean isReadyToPresent() {
            return this.state.get() == State.READY_TO_PRESENT;
        }

        public void copyTo(GpuDevice device, GpuTexture texture) {
            this.checkState(State.READY_TO_PRESENT);
            GpuExtensions.copyBufferToTexture(this.buffer, texture, 0, 0, this.frameSize.width(), this.frameSize.height(), NativeImage.Format.RGBA);
            if (this.recycleFence != null) {
                this.recycleFence.close();
            }
            this.recycleFence = device.createCommandEncoder().createFence();
        }

        public void recycle(GpuDevice device, FrameSize requestedFrameSize) {
            this.updateState(State.READY_TO_PRESENT, State.RECYCLING);
            this.tickRecycle(device, requestedFrameSize);
        }

        public void discardAndFreeze(GpuDevice device, FrameSize requestedFrameSize) {
            State state = this.state.get();
            if (state == State.CLOSED) {
                return;
            }
            if (state == State.WRITE_BEGIN) {
                throw new IllegalStateException("Cannot discard while write is in progress");
            }
            if (this.recycleFence != null) {
                this.recycleFence.awaitCompletion(Long.MAX_VALUE);
                this.recycleFence = null;
            }
            if (this.mappedView == null) {
                this.mapBuffer(device, requestedFrameSize);
            }
            this.updateState(state, State.FROZEN);
        }

        public void unfreeze() {
            this.updateState(State.FROZEN, State.READY_FOR_WRITE);
        }

        @Override
        public void close() {
            State state = this.state.get();
            if (state == State.CLOSED) {
                return;
            }
            while (state == State.WRITE_BEGIN || !this.state.compareAndSet(state, State.CLOSED)) {
                Thread.yield();
                state = this.state.get();
            }
            GpuBuffer.MappedView mappedView = this.mappedView;
            this.mappedView = null;
            if (mappedView != null) {
                mappedView.close();
            }
            if (this.recycleFence != null) {
                this.recycleFence.close();
                this.recycleFence = null;
            }
            this.buffer.close();
        }

        private static enum State {
            READY_FOR_WRITE,
            WRITE_BEGIN,
            WRITE_END,
            READY_TO_PRESENT,
            RECYCLING,
            FROZEN,
            CLOSED;

        }
    }
}

